<?php

namespace App\Http\Controllers;
use App\Models\Finance;
use App\Models\Reservation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
class FinanceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');

        $finances = Finance::with('reservation')
            ->when($search, function ($query, $search) {
                $query->where('type', 'like', "%$search%")
                    ->orWhere('source', 'like', "%$search%")
                    ->orWhere('description', 'like', "%$search%");
            })
            ->orderByDesc('date_operation')
            ->paginate(15);

        return view('finances.index', [
            'finances' => $finances,
            'search' => $search,
            'title' => 'Liste des opérations financières'
        ]);
    }

    public function traite(Request $request)
    {
        $search = $request->input('search');

        $finances = Finance::where('valide', 1)
            ->when('reservation')
            ->when($search, function ($query, $search) {
                $query->where('type', 'like', "%$search%")
                    ->orWhere('source', 'like', "%$search%")
                    ->orWhere('description', 'like', "%$search%");
            })
            ->orderByDesc('date_operation')
            ->paginate(15);

        return view('finances.index', [
            'finances' => $finances,
            'search' => $search,
            'title' => 'Liste des opérations financières'
        ]);
    }

    public function entre(Request $request)
    {
        $search = $request->input('search');

        $finances = Finance::where('sens', "entrée")
            ->when('reservation')
            ->when($search, function ($query, $search) {
                $query->where('type', 'like', "%$search%")
                    ->orWhere('source', 'like', "%$search%")
                    ->orWhere('description', 'like', "%$search%");
            })
            ->orderByDesc('date_operation')
            ->paginate(15);

        return view('finances.index', [
            'finances' => $finances,
            'search' => $search,
            'title' => 'Liste des opérations financières'
        ]);
    }

     public function sorti(Request $request)
    {
        $search = $request->input('search');

        $finances = Finance::where('sens', "sortie")
            ->when('reservation')
            ->when($search, function ($query, $search) {
                $query->where('type', 'like', "%$search%")
                    ->orWhere('source', 'like', "%$search%")
                    ->orWhere('description', 'like', "%$search%");
            })
            ->orderByDesc('date_operation')
            ->paginate(15);

        return view('finances.index', [
            'finances' => $finances,
            'search' => $search,
            'title' => 'Liste des opérations financières'
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $reservations = Reservation::where('statut', 1)
        ->orderByDesc('created_at')
        ->get(['id', 'code', 'depart', 'destination']);

        return view('finances.create', [
            'reservations' => $reservations,
            'title' => 'Nouvelle opération financière',
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'reservation_id' => 'required|exists:reservation,id',
            'type'           => 'required|in:paiement_client,commission_chauffeur,remboursement,bonus,autre',
            'montant'        => 'required|numeric|min:0',
            'source'         => 'nullable|string|max:255',
            'description'    => 'nullable|string',
            'valide'         => 'required|boolean',
            'sens'           => 'required|in:entrée,sortie',
        ]);

        Finance::create([
            'reservation_id' => $validated['reservation_id'],
            'type'           => $validated['type'],
            'montant'        => $validated['montant'],
            'source'         => $validated['source'] ?? null,
            'description'    => $validated['description'] ?? null,
            'valide'         => $validated['valide'],
            'sens'           => $validated['sens'],
            'date_operation' => now(),
        ]);

        return redirect()->route('finances.index')->with('success', 'Opération financière enregistrée avec succès.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $finance = Finance::findOrFail($id);

        $reservations = Reservation::where('statut', 1)
        ->orderByDesc('created_at')
        ->get(['id', 'code', 'depart', 'destination']);

        return view('finances.edit', [
            'reservations' => $reservations,
            'finance' => $finance,
            'title' => 'Edition opération financière',
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $finance = Finance::findOrFail($id);
        
        $validated = $request->validate([
            'sens' => 'required|in:entrée,sortie',
            'reservation_id' => 'required|exists:reservation,id',
            'type' => 'required|in:paiement_client,commission_chauffeur,remboursement,bonus,autre',
            'montant' => 'required|numeric|min:0',
            'source' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'valide' => 'required|boolean',
        ]);
    
        $finance->update($validated);
    
        return redirect()->route('finances.index')
            ->with('success', 'L’opération financière a été mise à jour avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    public function stat (){
        // Total des finances
            $totalFinanceValide = Finance::where('valide', 1)->sum('montant');
            $totalFinanceNonValide = Finance::where('valide', 0)->sum('montant');

            // Entrées / Sorties valides
            $totalEntreeValide = Finance::where('valide', 1)->where('sens', 'entrée')->sum('montant');
            $totalSortieValide = Finance::where('valide', 1)->where('sens', 'sortie')->sum('montant');
            $soldeValide = $totalEntreeValide - $totalSortieValide;

            // Entrées / Sorties NON validées
            $totalEntreeNonValide = Finance::where('valide', 0)->where('sens', 'entrée')->sum('montant');
            $totalSortieNonValide = Finance::where('valide', 0)->where('sens', 'sortie')->sum('montant');
            $soldeNonValide = $totalEntreeNonValide - $totalSortieNonValide;

            // Graph mensuel
            $mensuel = Finance::where('valide', 1)
                ->selectRaw("DATE_FORMAT(date_operation, '%Y-%m') as mois,
                    SUM(CASE WHEN sens = 'entrée' THEN montant ELSE 0 END) as revenus,
                    SUM(CASE WHEN sens = 'sortie' THEN montant ELSE 0 END) as depenses")
                ->groupBy('mois')
                ->orderBy('mois')
                ->get();

            // Graph par source
            $parSource = Finance::where('valide', 1)
                ->select('source', DB::raw('SUM(montant) as total'))
                ->groupBy('source')
                ->get();

            return view('finances.stat', compact(
                'totalFinanceValide',
                'totalFinanceNonValide',
                'totalEntreeValide',
                'totalSortieValide',
                'soldeValide',
                'mensuel',
                'parSource',
                'totalEntreeNonValide',
                'totalSortieNonValide',
                'soldeNonValide'
            ));
    }
}
