<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Quartier;
use App\Models\Prison;
use App\Models\Crime;
use App\Models\CrimePhoto;
use App\Models\CrimePrisonier;
use App\Models\Prisonier;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class CrimeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $crimes = Crime::orderBy('created_at', 'desc')->get();
        $title="Les crimes";
        return view('prisoniers.crimes.index', compact('crimes','title',));
    }
    function generateUniqueMatricule($length = 5)
    {
        do {
            $letters = strtoupper(Str::random(2)); // lettres A-Z
            $numbers = str_pad(random_int(0, 999), 3, '0', STR_PAD_LEFT); // nombres sur 4 chiffres
            $matricule = $letters . $numbers;
        } while (Crime::where('code', $matricule)->exists());
        return $matricule;
    }
    public function cc(Request $request)
    {
        $prisons = Prison::with(['quartier.arrondissement.zone.ville'])->where('type', 'CC')->get();

        $title="Commisairiats centraux";
        
        return view('prisons.index', compact('prisons',"title"));
    }
    public function cpa(Request $request)
    {
        $prisons = Prison::with(['quartier.arrondissement.zone.ville'])->where('type', 'CPA')->get();

        $title="Commisairiats des polices d'arrondissement";
        
        return view('prisons.index', compact('prisons',"title"));
    }
    public function cpq(Request $request)
    {
        $prisons = Prison::with(['quartier.arrondissement.zone.ville'])->where('type', 'CPQ')->get();

        $title="Commisairiats des polices de quartier";
        
        return view('prisons.index', compact('prisons',"title"));
    }
    public function pa(Request $request)
    {
        $prisons = Prison::with(['quartier.arrondissement.zone.ville'])->where('type', 'PA')->get();

        $title="Les postes avancés";
        
        return view('prisons.index', compact('prisons',"title"));
    }

    // fonction create
    public function create()
    {
         $quartiers = Quartier::all();
         $title="Ajouter un crime";
         $code=$this->generateUniqueMatricule();
        return view('prisoniers.crimes.create', compact('quartiers','title','code'));
    }

    public function store(Request $request)
    {
        // Validation des données
        $request->validate([
            'code' => 'required|unique:crimes,code|max:255',
            'libelle' => 'required|string|max:255',
            'description' => 'nullable|string',
            'niveau_gravite' => 'required|in:Faible,Moyenne,Élevée',
            'loi_associee' => 'nullable|string|max:255',
            'date_crime' => 'nullable|date',
            'lieu_crime' => 'nullable|string|max:255',
            'quartier_id' => 'required|exists:quartiers,id', // Validation que le quartier existe
        ]);

        // Création du crime
        $crime = Crime::create([
            'code' => $request->code,
            'libelle' => $request->libelle,
            'description' => $request->description,
            'niveau_gravite' => $request->niveau_gravite,
            'loi_associee' => $request->loi_associee,
            'date_crime' => $request->date_crime,
            'lieu_crime' => $request->lieu_crime,
            'quartier_id' => $request->quartier_id, // Enregistrement de la relation avec le quartier
        ]);

        // Retourner une réponse indiquant le succès de l'enregistrement
         return redirect()->route('crimes.index')->with('success', 'Crime enregistré avec succès.');
        
    }


    public function detail($id)
    {
        $crime = Crime::findOrFail($id);
        $quartiers = Quartier::all();
        // Récupérer les crimes et les prisonniers
        $prisonniers = Prisonier::all();

        return view('prisoniers.crimes.detail', compact('crime', 'quartiers','prisonniers'));
    }
    public function photo_store(Request $request)
    {
        // Validation des photos
        $request->validate([
            'photos' => 'required|array|max:5',  // Maximum 5 photos
            'photos.*' => 'image|mimes:jpg,png,jpeg,gif,svg|max:2048', // Format et taille des images
        ]);

       // dd($request->all());

        // Vérifier si des photos sont téléchargées
        if ($request->hasFile('photos')) {
            foreach ($request->file('photos') as $photo) {
                // Enregistrer la photo dans le dossier public/photos et obtenir son chemin
                $path = $photo->store('photos_crimes', 'public');

                // Créer un enregistrement dans la table crime_photo pour chaque photo
                CrimePhoto::create([
                    'crime_id' => $request->id_crime,  // Lier la photo au crime
                    'photo_path' => $path,     // Stocker le chemin de la photo
                ]);
            }
        }

        // Rediriger vers la page de détails du crime avec un message de succès
        return redirect()->route('crimes.detail', $request->id_crime)->with('success', 'Photos ajoutées avec succès!');
    
    }
    public function prisonier_store(Request $request)
    {
        // Validation des données
        $request->validate([
            'crime_id' => 'required|exists:crimes,id',
            'prisonnier_id' => 'required|exists:prisoniers,id',
            'rôle' => 'nullable|string|max:50',
            'date_implication' => 'nullable|date',
            'condamnation' => 'nullable|string',
        ]);

        // Création de la nouvelle relation entre crime et prisonnier
        CrimePrisonier::create([
            'crime_id' => $request->crime_id,
            'prisonnier_id' => $request->prisonnier_id,
            'rôle' => $request->rôle,
            'date_implication' => $request->date_implication,
            'condamnation' => $request->condamnation,
        ]);
        // Rediriger vers la page de détails du crime avec un message de succès
        return redirect()->route('crimes.detail', $request->crime_id)->with('success', 'Photos ajoutées avec succès!');
    
    }
    public function edit($id)
    {
        $crime = Crime::findOrFail($id);
        $quartiers = Quartier::all();
        return view('prisoniers.crimes.edit', compact('crime', 'quartiers'));
    }

    public function update(Request $request, $id)
    {
        // Trouver le crime à mettre à jour
        $crime = Crime::findOrFail($id);

        // Validation des données envoyées
        $request->validate([
            'libelle' => 'required|string|max:255', // Validation du libellé
            'niveau_gravite' => 'required|in:Faible,Moyenne,Élevée', // Validation du niveau de gravité
            'loi_associee' => 'nullable|string|max:255', // Validation de la loi associée
            'date_crime' => 'nullable|date', // Validation de la date
            'lieu_crime' => 'nullable|string|max:255', // Validation du lieu
            'quartier_id' => 'required|exists:quartiers,id', // Validation du quartier (il doit exister dans la table `quartiers`)
            'description' => 'nullable|string', // Validation de la description
        ]);

        // Mise à jour des données du crime avec les nouvelles informations
        $crime->update([
            'libelle' => $request->libelle,
            'niveau_gravite' => $request->niveau_gravite,
            'loi_associee' => $request->loi_associee,
            'date_crime' => $request->date_crime,
            'lieu_crime' => $request->lieu_crime,
            'quartier_id' => $request->quartier_id,
            'description' => $request->description,
        ]);

        // Si une nouvelle photo est envoyée, on la gère ici
        if ($request->hasFile('photo')) {
            // Supprimer l'ancienne photo, si elle existe
            if ($crime->photo && Storage::exists('public/' . $crime->photo)) {
                Storage::delete('public/' . $crime->photo);
            }

            // Sauvegarder la nouvelle photo et récupérer son chemin
            $path = $request->file('photo')->store('photos', 'public');
            $crime->photo = $path;
            $crime->save(); // Sauvegarder la photo dans la base de données
        }

        // Rediriger avec un message de succès
        return redirect()->route('crimes.index')->with('success', 'Crime mis à jour avec succès.');
    }


    public function stat()
    {
        $totalPrisons = Prison::count();

        $prisonsByType = Prison::select('type', DB::raw('count(*) as total'))
            ->groupBy('type')
            ->pluck('total', 'type');

        $topQuartier = Prison::join('quartiers', 'prisons.quartier_id', '=', 'quartiers.id')
            ->select('quartiers.name', DB::raw('count(*) as total'))
            ->groupBy('quartiers.name')
            ->orderByDesc('total')
            ->first();

        $largestPrison = Prison::orderByDesc('capacity')->first();

        $prisonsByStatus = Prison::select('is_active', DB::raw('count(*) as total'))
            ->groupBy('is_active')
            ->pluck('total', 'is_active');

            $title="Les statistiques";

            //dd($prisonsByType);

        return view('prisons.stat', compact(
            'totalPrisons', 'prisonsByType', 'topQuartier', 'largestPrison', 'prisonsByStatus','title'
        ));
    }



}
